unit uSMS;

          {ͻ}
          {                                                        }
          {  unit uSMS                                             }
          {                                                        }
          {  jednotka s definicemi pro SMS zpravy pres GSM modem   }
          {                                                        }
          {  (C)1999 SofCon, Steovick 49, 160 00 Praha 6        }
          {               Adam Wild, Dejvicka 42, Praha 6          }
          {ͼ}

{--------}
interface
{--------}

(* Terminology:
     SM   - Short Message
     MS   - Mobile Station
     SME  - Short Message Entity
     SMSC - Short Message Service Center
     MMI  - Man Machine Units
     PDU  - Protocol Data Unit
     TE   - Terminal (PC)
*)

type
  tStringSMS   = string[160]; { typ retezce pro vlastni text SMS zpravy }
  tStringPDU   = string[255]; { typ retezce pro text SMS zpravy v PDU modu }
  tPhoneNumStr = string[16];  { typ retezce pro telefonni cislo }

  tTypTime =  { vycet ruznych formatu casu }
    (tmAbsTime,   { absolutni cas }
     tmRelTime,   { relativni cas }
     tmNoTime);   { cas neni uveden }

  tMsgStat =  { vycet zakladniho statusu zpravy }
    (tsRecUnRead, { prijata a jeste neprectena (nova) zprava }
     tsRecRead,   { prijata a jiz prectena (stara) zprava }
     tsStoUnSent, { zprava pro vyslani, ale jeste nevyslana }
     tsStoSent,   { zprava pro vyslani a jiz vyslana }
     tsAll);      { jakakoliv zprava }

type
 { struktura cele SMS zpravy }
  pSendRecord = ^tSendRecord;
  tSendRecord = record
    MsgStat      : tMsgStat;     { zakladni status zpravy }
    PDU_Type     : byte;         { typ PDU zpravy }
    MsgReference : byte;         {!pouze pro vysilanou zpravu }
    PhoneNumType : byte;         { typ telefonniho cisla }
    PhoneNum     : tPhoneNumStr; { telefonni cislo (Originator Adress/Destination Adress) }
    Protocol_ID  : byte;         { Protocol Identifier }
    DataCodScheme: byte;         { Data Coding Scheme }
    SMS_Message  : tStringSMS;   { vlastni zprava }
    case TypTime : tTypTime of
      tmAbsTime:
        (Time    : tDateTime;    { absolutni datum a cas }
         TimeZone: ShortInt;     { posunuti oproti svetovemu casu }
        );
      tmRelTime:
        (Mins    : word;         { relativni pocet minut }
         Days    : word;         { relativni pocet dnu }
        );
      tmNoTime:
        ();
  end;

  pRecRecord = ^tRecRecord;
  tRecRecord = tSendRecord;

const
 { konstanty typu telefonnich cisel }
  cPNT_National = $81; { telefonni cislo je narodni }
  cPNT_Internat = $91; { telefonni cislo je mezinarodni }
  { casem zde mohou pribyt dalsi konstanty napr. pro WWW apod }

 { konstanty pro Protocol_ID }
  cPID_SMS      = $00; { zprava je SMS }
  cPID_Telex    = $01; { zprava je telex }
  cPID_TeleFax3 = $02; { zprava je group3 telefax }
  cPID_Telefax4 = $03; { zprava je group4 telefax }
  { casem zde mohou pribyt dalsi konstanty }

 { konstanty pro DataCodScheme }
  cDCS_Impl7bit = $0000; { implicitni (7-mi bitove) kodovani textu zpravy }
  { casem zde mohou pribyt dalsi konstanty }

(*
 - PDU_Type
   ~~~~~~~~
     case PDU_Type and $03 of

     00b:
    {===== zprava v prijimacim formatu =====}
    { pouziva se pro prijatou zpravu od SMSC do MS nebo pro pripadnou odpoved
      od MS do SMSC na prijatou zpravu }
     begin
                             7    6    5    4    3    2
                          -------------------------------
       PDU_Type and $FC = | RP |UDHI| SRI|  0 |  0 | MMS|
                          -------------------------------
        RP   (Reply Path)
          - bit indikuje, ze existuje "Reply Path"                      - {Nevim, co je Reply Path}
        UDHI (User Data Header Indicator)
          - bit indikuje, ze textove pole SMS_Message obsahuje hlavicku - {Nevim, jakou hlavicku}
        SRI  (Status Report Indication)
          - bit indikuje, ze SME prijalo status odpoved od SMSC
        MMS  (More Messages to Send)
          - bit indikuje, ze neni dalsich zprav                         - {Nevim, jakych dalsich}

       {a dale zprava pokracuje}
          1    1   LenPN    1   1    7    0 - 140
       ---------------------------------------------
       |LenPN|PNT|PhoneNum|PID|DCS|Time|SMS_Message|
       ---------------------------------------------
        LenPN = delka telefonniho cisla PhoneNum
        PNT = PhoneNumType
              - 81h - telefonni cislo je narodni
              - 91h - telefonni cislo je mezinarodni
              - jine (napr. pro WWW apod)
        PID = Protocol_ID
              - 00h - SMS
              - 01h - telex
              - 02h - group3 telefax
              - 03h - group4 telefax
              - jine
        DCS = DataCodScheme
              - 0000 0000b - implicitni (7-mi bitove) kodovani textu zpravy
              - 0001 1111b - rezervovano
              - 0001 1110b - rezervovano
              - 1111 XXXXb
                     ||\|
                     || --- 00 - Class0 okamzite zobrazitelne
                     ||     01 - Class1 ME (Mobile Equipment) - specificke
                     ||     10 - Class2 SIM specificka zprava
                     ||     11 - Class3 TE (Terminate Equipment) - specificke
                     | ---- 0  - implicitni (7-mi bitove) kodovani textu zpravy
                     |      1  - 8-mi bitove kodovani textu zpravy
                      ----- rezervovan (0)
        delka SMS_Message je v zapakovanem tvaru
     end;

     01b:
    {===== zprava ve vysilacim formatu =====}
    { pouziva se pro vysilanou zpravu od MS do SMSC nebo pro prijatou odpoved
      od SMSC do MS na vyslanou zpravu }
     begin
                             7    6    5    4    3    2
                          -------------------------------
       PDU_Type and $FC = | RP |UDHI| SRR|   VPF   | RD |
                          -------------------------------
        RP   (Reply Path) - viz vyse
        UDHI (User Data Header Indicator) - viz vyse
        SRR  (Status Report Request)
          - bit indikuje, ze MS pozaduje status odpoved od SMSC
        VPF  (Validity Period Format)
          - 00 zprava neobsahuje pole Time ani TimeZone
            01 rezervovano
            10 zprava obsahuje pole Time, kde Time je relativni cas
            11 zprava obsahuje pole Time a TimeZone, kde Time je absolutni cas
        RD   (Reject Duplicate)                                         - {Nevim, co to je}

       {a dale zprava pokracuje}
         1   1    1   LenPN    1   1  0,1,7   0 - 140
       ------------------------------------------------
       |MR|LenPN|PNT|PhoneNum|PID|DCS|Time|SMS_Message|
       ------------------------------------------------
        MR  = MsgReference  - {Nevim co to je}
        LenPN = delka telefonniho cisla PhoneNum
        PNT = PhoneNumType  - viz vyse
        PID = Protocol_ID   - viz vyse
        DCS = DataCodScheme - viz vyse
        delka SMS_Message je v zapakovanem tvaru
        if VPF = 11 then Time viz vyse
        if VPF = 10 then Time je relativni cas a to:
           00:05  az 12:00  (step  5min) (VP=  0..143)
           12:30  az 24:00  (step 30min) (VP=144..167)
           2days  az 30days (step 1day ) (VP=168..196)
           35days az 441days(step 7days) (VP=197..255)
     end;

     10b:
    {===== =====}
    { pouziva se pro vyslani statusu na prikaz od SMSC do MS nebo pro vyslani
      prikazu od MS do SMSC }

     11b:
    {===== rezervovano =====}
     end;

*)

function GetSMS_PDUStr(const S:tStringPDU):tStringSMS;
  { funkce prevede retezec znaku SMS zpravy v PDU modu na retezec znaku ASCII }

function GetPDU_SMSStr(const S:tStringSMS):tStringPDU;
  { funkce prevede retezec znaku SMS zpravy v ASCII na retezec znaku v PDU modu }

procedure DecodeRelTime(B:byte; var Mins,Days:word);
  { rozdekoduje relativni cas zakodovany v B a vyplni polozky Mins a Days }
function  CodeRelTime(Mins,Days:word):byte;
  { zakoduje relativni cas v Mins ci Days do bytu, ktery vrati jako funkcni hodnotu }

function DecodeRecMsg(RMess:pRecRecord; const S:string):word;
  { funkce dekoduje celou prijatou SMS zpravu do bufferu a vrati jeho delku }

function CodeSendMsg(SMess:pSendRecord):string;
  { funkce koduje zpravu v zaznamu vysilaciho recordu SMess do vysilaciho stringu }

{-------------}
implementation
{-------------}

uses SysUtils;

{-------------------------}
function GetSMS_PDUStr(const S:tStringPDU):tStringSMS;
var
  ResS:tStringSMS;
  I   :byte;
  BNib:boolean;
  B,BB:byte;
  NumB:0..8;
  SavB:byte;
begin
  ResS:='';
  BNib:=False;
  NumB:=0;
  SavB:=0;
  B   :=0;
  for I:=1 to Length(S) do
    begin
     { precteni jednoho ASCII znaku a prepocet na Nibble }
      BB:=StrToInt('$'+S[I]);
     { prevod dvou Nibble na Byte }
      if not BNib then B:=BB
                  else B:=(B shl 4) or BB;
      BNib:=not BNib;
     { vlastni prevod 7mi bitoveho bytu na znak }
      if not BNib then
      begin
        if NumB=0 then
        begin
          ResS:=ResS+Chr(B and $7F);
          NumB:=1;
          SavB:=B;
        end
        else
        begin
          ResS:=ResS+Chr(((B shl NumB) or (SavB shr (8-NumB))) and $7F);
          Inc(NumB);
          SavB:=B;
          if NumB=7 then
            begin
              if (SavB shr 1)<>0 then
                ResS:=ResS+Chr((SavB shr 1) and $7F);
              NumB:=0;
            end;
        end;
      end;
    end;
  GetSMS_PDUStr:=ResS;
end;

{-------------------------}
function GetPDU_SMSStr(const S:tStringSMS):tStringPDU;
const
  Mask:array[0..7]of byte =($01,$03,$07,$0F,$1F,$3F,$7F,$FF);
var
  ResS:tStringPDU;
  I   :byte;
  NumB:byte;
  B   :byte;
label
  L_Break;
begin
  ResS:='';
  NumB:=0;
  I:=1;
  while I<=Length(S) do
  begin
    if Length(S)>I then
         B:=(Ord(S[I]) shr NumB) or ((Ord(S[I+1]) and Mask[NumB]) shl (7-NumB))
    else B:= Ord(S[I]) shr NumB;
    Inc(NumB);
    if NumB=7 then
    begin
      NumB:=0;
      Inc(I);
    end;
    ResS:=ResS+IntToHex((B and $F0) shr 4,1)+IntToHex(B and $0F,1);
    if I>=Length(S) then goto L_Break;
    Inc(I);
  end;
 L_Break:
  GetPDU_SMSStr:=ResS;
end;

{-------------------------}
procedure DecodeRelTime(B:byte; var Mins,Days:word);
begin
  Mins:=0;
  Days:=0;
  case B of
      0..143: Mins:=(B+1)*5;
    144..166: Mins:=720+(B-143)*30;
    167     : Days:=1;
    168..196: Days:=(B-166)*1;
    197..255: Days:=(B-192)*7;
  end;
end;

{-------------------------}
function  CodeRelTime(Mins,Days:word):byte;
var B:byte;
begin
  if Days<>0 then
  case Days of
     1     :B:=167;
     2.. 30:B:=Days+166;
    31.. 34:B:=196;
    35..441:B:=(Days div 7)+192;
    else    B:=255;
  end
  else
  case Mins of
      0..   9:B:=0;
     10.. 724:B:=(Mins div 5)-1;
    725.. 749:B:=143;
    750..1439:B:=((Mins-720) div 30)+143;
    else      B:=167;
  end;
  CodeRelTime:=B;
end;

{-------------------------}
type
  tCtrl =
    (tpPDU_Type,
     tpMR,
     tpPNT,
     tpLenPhoneNum,
     tpPhoneNum,
     tpPID,
     tpDCS,
     tpAbsTime_Year,
     tpAbsTime_Month,
     tpAbsTime_Day,
     tpAbsTime_Hour,
     tpAbsTime_Min,
     tpAbsTime_Sec,
     tpAbsTime_Zone,
     tpRelTime,
     tpSMS);

function DecodeRecMsg(RMess:pRecRecord; const S:string):word;
var
  I,B : byte;
  Ctrl: tCtrl;
  Len : word;
  LPN : byte;
  Ch  : char;
  FlFirst : boolean;
  Year, Month, Day,
  Hour, Min,   Sec : word;
label
  L_End;
begin
  I:=1;
  Len:=0;
  LPN:=0;
  Year:=0; Month:=0; Day:=0;
  Hour:=0; Min  :=0; Sec:=0;
  Ctrl:=tpPDU_Type;
  FlFirst:=True;

  with RMess^ do
    while I<=Length(S) do
    begin
      case Ctrl of
        tpPDU_Type:
          if FlFirst then
          begin
            B:=StrToInt('$'+S[I]) shl 4;
            FlFirst:=False;
          end
          else
          begin
            PDU_Type:=B or StrToInt('$'+S[I]);
            FlFirst:=True;
            case (PDU_Type and $03) of
              $00:
                begin
                  MsgReference:=0;
                  Ctrl:=tpLenPhoneNum;
                end;
              $01:
                begin
                  Ctrl:=tpMR;
                end;
              $02,
              $03:
                begin
                  goto L_End;
                end;
            end;
          end;
        tpMR:
          if FlFirst then
          begin
            B:=StrToInt('$'+S[I]) shl 4;
            FlFirst:=False;
          end
          else
          begin
            MsgReference:=B or StrToInt('$'+S[I]);
            FlFirst:=True;
            Ctrl:=tpLenPhoneNum;
          end;
        tpLenPhoneNum:
          if FlFirst then
          begin
            B:=StrToInt('$'+S[I]) shl 4;
            FlFirst:=False;
          end
          else
          begin
            FlFirst:=True;
            LPN:=B or StrToInt('$'+S[I]);
            if (LPN mod 2)<>0 then Inc(LPN);
            Ctrl:=tpPNT;
          end;
        tpPNT:
          if FlFirst then
          begin
            B:=StrToInt('$'+S[I]) shl 4;
            FlFirst:=False;
          end
          else
          begin
            FlFirst:=True;
            PhoneNumType:=B or StrToInt('$'+S[I]);
            PhoneNum:='';
            if LPN>0 then
                 Ctrl:=tpPhoneNum
            else Ctrl:=tpPID;
          end;
        tpPhoneNum:
          begin
            PhoneNum:=PhoneNum+S[I];
            if Length(PhoneNum)=LPN then
            begin
              for B:=1 to Length(PhoneNum) do
                if (B mod 2)=0 then
                begin
                  Ch:=PhoneNum[B-1];
                  PhoneNum[B-1]:=PhoneNum[B];
                  PhoneNum[B]:=Ch;
                end;
              while UpCase(PhoneNum[Length(PhoneNum)])='F' do
                Delete(PhoneNum,Length(PhoneNum),1);
              Ctrl:=tpPID;
            end;
          end;
        tpPID:
          if FlFirst then
          begin
            B:=StrToInt('$'+S[I]) shl 4;
            FlFirst:=False;
          end
          else
          begin
            Protocol_ID:=B or StrToInt('$'+S[I]);
            FlFirst:=True;
            Ctrl:=tpDCS;
          end;
        tpDCS:
          if FlFirst then
          begin
            B:=StrToInt('$'+S[I]) shl 4;
            FlFirst:=False;
          end
          else
          begin
            DataCodScheme:=B or StrToInt('$'+S[I]);
            FlFirst:=True;
            if (PDU_Type and $03)=$00 then
              begin
                TypTime:=tmAbsTime;
                Ctrl:=tpAbsTime_Year;
              end;
            if (PDU_Type and $03)=$01 then
              case (PDU_Type and $18) shr 3 of
                $00:begin
                      TypTime:=tmNoTime;
                      Ctrl:=tpSMS;
                    end;
                $01:begin
                      TypTime:=tmNoTime;
                      goto L_End;
                    end;
                $02:begin
                      TypTime:=tmRelTime;
                      Ctrl:=tpRelTime;
                    end;
                $03:begin
                      TypTime:=tmAbsTime;
                      Ctrl:=tpAbsTime_Year;
                    end;
              end;
          end;
        tpAbsTime_Year:
          if FlFirst then
          begin
            B:=StrToIntDef(S[I],0);
            FlFirst:=False;
          end
          else
          begin
            B:=B + (StrToIntDef(S[I],0)*10);
            if B<80 then
                 Year:=2000+B
            else Year:=1900+B;
            FlFirst:=True;
            Ctrl:=tpAbsTime_Month;
          end;
        tpAbsTime_Month:
          if FlFirst then
          begin
            B:=StrToIntDef(S[I],0);
            FlFirst:=False;
          end
          else
          begin
            B:=B + (StrToIntDef(S[I],0)*10);
            Month:=B;
            FlFirst:=True;
            Ctrl:=tpAbsTime_Day;
          end;
        tpAbsTime_Day:
          if FlFirst then
          begin
            B:=StrToIntDef(S[I],0);
            FlFirst:=False;
          end
          else
          begin
            B:=B + (StrToIntDef(S[I],0)*10);
            Day:=B;
            FlFirst:=True;
            Ctrl:=tpAbsTime_Hour;
          end;
        tpAbsTime_Hour:
          if FlFirst then
          begin
            B:=StrToIntDef(S[I],0);
            FlFirst:=False;
          end
          else
          begin
            B:=B + (StrToIntDef(S[I],0)*10);
            Hour:=B;
            FlFirst:=True;
            Ctrl:=tpAbsTime_Min;
          end;
        tpAbsTime_Min:
          if FlFirst then
          begin
            B:=StrToIntDef(S[I],0);
            FlFirst:=False;
          end
          else
          begin
            B:=B + (StrToIntDef(S[I],0)*10);
            Min:=B;
            FlFirst:=True;
            Ctrl:=tpAbsTime_Sec;
          end;
        tpAbsTime_Sec:
          if FlFirst then
          begin
            B:=StrToIntDef(S[I],0);
            FlFirst:=False;
          end
          else
          begin
            B:=B + (StrToIntDef(S[I],0)*10);
            Sec:=B;
            Time:=EncodeDate(Year,Month,Day)+EncodeTime(Hour,Min,Sec,0);
            FlFirst:=True;
            Ctrl:=tpAbsTime_Zone;
          end;
        tpAbsTime_Zone:
          if FlFirst then
          begin
            B:=StrToIntDef(S[I],0);
            FlFirst:=False;
          end
          else
          begin
            B:=B + (StrToIntDef(S[I],0)*10);
            TimeZone:=B;
            FlFirst:=True;
            Ctrl:=tpSMS;
          end;
        tpRelTime:
          if FlFirst then
          begin
            B:=StrToIntDef(S[I],0);
            FlFirst:=False;
          end
          else
          begin
            B:=B + (StrToIntDef(S[I],0)*10);
            DecodeRelTime(B,Mins,Days);
            FlFirst:=True;
            Ctrl:=tpSMS;
          end;
        tpSMS:
          if FlFirst then
          begin
            B:=StrToInt('$'+S[I]) shl 4;
            FlFirst:=False;
          end
          else
          begin
            {B:=B or StrToInt('$'+S[I]);}
            SMS_Message:=GetSMS_PDUStr(Copy(S,I+1,Length(S)));
            Len:=SizeOf(PDU_Type)     +
                 SizeOf(MsgReference) +
                 SizeOf(PhoneNumType) +
                 SizeOf(PhoneNum)     +
                 SizeOf(Protocol_ID)  +
                 SizeOf(DataCodScheme)+
                 SizeOf(Time)         +
                 SizeOf(TimeZone)     +
                 +1+Length(SMS_Message);
            goto L_End;
          end;
      end;
      Inc(I);
    end;
 L_End:
  DecodeRecMsg:=Len;
end;

{-------------------------}
function CodeSendMsg(SMess:pSendRecord):string;
var
  S : string;
  {----------}
  procedure SendBStrHex(B: byte);
    { zakodovani bytu jako string s hex representaci B do vysilaciho bufferu }
  begin
    S:=S+IntToHex(B,2);
  end;
  {----------}
  procedure SendPhoneNum(PhoneNum: tPhoneNumStr);
    { zakodovani telefonniho cisla do vysilaciho bufferu }
  var I :byte;
      Ch:char;
  begin
    for I:=1 to Length(PhoneNum) do
      begin
        if (I mod 2)=0 then
        begin
          Ch:=PhoneNum[I-1];
          PhoneNum[I-1]:=PhoneNum[I];
          PhoneNum[I]:=Ch;
        end;
      end;
    if (Length(PhoneNum) mod 2)<>0 then
      begin
        Ch:=PhoneNum[Length(PhoneNum)];
        PhoneNum[Length(PhoneNum)]:='F';
        PhoneNum:=PhoneNum+Ch;
      end;
    S:=S+PhoneNum;
  end;
  {----------}
var
  I   :byte;
  Ch  :char;
  PomS:string;
begin
  S:='';
  with SMess^ do
  begin
   { PDU_Type }
    SendBStrHex(PDU_Type);
   { (MR) }
    if (PDU_Type and $03)=$01 then
      SendBStrHex(MsgReference);
   { LenPN }
    SendBStrHex(Length(PhoneNum));
   { PNT }
    SendBStrHex(PhoneNumType);
   { (PhoneNum) }
    if Length(PhoneNum)<>0 then
      SendPhoneNum(PhoneNum);
   { PID }
    SendBStrHex(Protocol_ID);
   { DCS }
    SendBStrHex(DataCodScheme);
   { (Time) }
    case TypTime of
      tmAbsTime:
        begin
          DateTimeToString(PomS,'yymmdd',Time);
          for I:=1 to Length(PomS) do
            if (I mod 2)=0 then
            begin
              Ch:=PomS[I-1];
              PomS[I-1]:=PomS[I];
              PomS[I]:=Ch;
            end;
          S:=S+PomS;
          DateTimeToString(PomS,'hhnnss',Time);
          for I:=1 to Length(PomS) do
            if (I mod 2)=0 then
            begin
              Ch:=S[I-1];
              PomS[I-1]:=PomS[I];
              PomS[I]:=Ch;
            end;
          S:=S+PomS;
          SendBStrHex(TimeZone);
        end;
      tmRelTime:
        begin
          SendBStrHex(CodeRelTime(Mins,Days));
        end;
    end;
   { Len SMS_Message }
    SendBStrHex(Length(SMS_Message));
   { SMS_Message }
    S:=S+GetPDU_SMSStr(SMS_Message);
   { Ctrl+Z }
    S:=S+Chr($1A);
  end;
  CodeSendMsg:=S;
end;

{-------------------------}
End.
